// src/utils/runtimeConfig.ts
var DEFAULT_API_ORIGIN = "http://localhost:8000";
function normalizeOrigin(origin) {
  return origin.replace(/\/+$/, "");
}
var injectedOrigin = true ? "https://grovex.space".trim() : "";
var SERVER_BASE_URL = normalizeOrigin(injectedOrigin || DEFAULT_API_ORIGIN);
var API_BASE_URL = `${SERVER_BASE_URL}/api`;

// src/background/index.ts
function getRandomDelay(settings) {
  const min = settings.minWaitTime ?? settings.timingBetweenPrompts ?? 3;
  const max = settings.maxWaitTime ?? min + 5;
  const clamped = Math.max(min, 1);
  return (clamped + Math.random() * Math.max(0, max - clamped)) * 1e3;
}
var state = {
  prompts: [],
  settings: null,
  isRunning: false,
  isPaused: false,
  activeSlots: /* @__PURE__ */ new Map(),
  lastResultUrl: null,
  generationMode: "single"
};
async function trackUsageOnServer(service) {
  try {
    const { accessToken, user } = await chrome.storage.local.get(["accessToken", "user"]);
    if (!accessToken) {
      sendLog("warning", "Not logged in - usage not tracked");
      return;
    }
    const res = await fetch(`${API_BASE_URL}/usage/track`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "Authorization": `Bearer ${accessToken}`
      },
      body: JSON.stringify({ service, prompt_count: 1 })
    });
    if (res.ok) {
      const data = await res.json();
      if (user) {
        const updatedUser = {
          ...user,
          daily_generations: (user.daily_generations || 0) + 1
        };
        await chrome.storage.local.set({ user: updatedUser });
        chrome.runtime.sendMessage({
          type: "USAGE_UPDATED",
          payload: {
            remaining: data.remaining,
            used: updatedUser.daily_generations
          }
        }).catch(() => {
        });
      }
      sendLog("info", `Credit used. Remaining: ${data.remaining}`);
    } else {
      const error = await res.json().catch(() => ({ detail: "Unknown error" }));
      sendLog("warning", `Usage track failed: ${error.detail || res.status}`);
    }
  } catch (e) {
    console.warn("[GenFlow] Usage track error:", e);
    sendLog("warning", `Usage track error: ${e}`);
  }
}
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case "START_GENERATION":
      handleStartGeneration(message.payload);
      sendResponse({ success: true });
      break;
    case "PAUSE_GENERATION":
      state.isPaused = true;
      sendResponse({ success: true });
      break;
    case "RESUME_GENERATION":
      state.isPaused = false;
      processNextPrompt();
      sendResponse({ success: true });
      break;
    case "STOP_GENERATION":
      handleStopGeneration();
      sendResponse({ success: true });
      break;
    case "RETRY_FAILED":
      handleRetryFailed();
      sendResponse({ success: true });
      break;
    case "GET_STATUS":
      sendResponse({
        isRunning: state.isRunning,
        isPaused: state.isPaused,
        activeSlots: state.activeSlots.size,
        pendingCount: state.prompts.filter((p) => p.status === "pending").length
      });
      break;
    case "GET_STATE":
      sendResponse({
        success: true,
        payload: {
          isRunning: state.isRunning,
          isPaused: state.isPaused,
          activeSlots: state.activeSlots.size,
          prompts: state.prompts
        }
      });
      break;
    case "GENERATION_COMPLETE":
      handleGenerationComplete(message.payload);
      sendResponse({ success: true });
      break;
    case "GENERATION_FAILED":
      handleGenerationFailed(message.payload);
      sendResponse({ success: true });
      break;
    case "POLICY_ERROR":
      handlePolicyError(message.payload);
      sendResponse({ success: true });
      break;
    case "DOWNLOAD_RESULT":
      handleDownload(message.payload);
      sendResponse({ success: true });
      break;
    default:
      sendResponse({ success: false, error: "Unknown message type" });
  }
  return true;
});
function expandPromptsForGenerations(prompts, settings) {
  const perPrompt = settings.generationsPerPrompt || 1;
  if (perPrompt <= 1)
    return prompts;
  const expanded = [];
  for (const prompt of prompts) {
    for (let i = 0; i < perPrompt; i++) {
      expanded.push({
        ...prompt,
        id: i === 0 ? prompt.id : crypto.randomUUID(),
        number: prompt.number
      });
    }
  }
  return expanded;
}
function handlePolicyError(payload) {
  const prompt = state.prompts.find((p) => p.id === payload.promptId);
  if (!prompt)
    return;
  updatePromptStatus(payload.promptId, "failed", {
    error: "Policy violation - prompt skipped"
  });
  sendLog("warning", `Prompt #${prompt.number} skipped (policy violation)`);
  for (const [slotId, slot] of state.activeSlots.entries()) {
    if (slot.promptId === payload.promptId) {
      state.activeSlots.delete(slotId);
      break;
    }
  }
  const delay = state.settings ? getRandomDelay(state.settings) : 3e3;
  setTimeout(() => processNextPrompt(), delay);
}
async function handleStartGeneration(payload) {
  state.settings = payload.settings;
  state.generationMode = payload.generationMode || "single";
  state.isRunning = true;
  state.isPaused = false;
  state.activeSlots.clear();
  state.lastResultUrl = null;
  const startFrom = (payload.settings.startFromPrompt || 1) - 1;
  const sliced = payload.prompts.slice(startFrom);
  state.prompts = expandPromptsForGenerations(sliced, payload.settings);
  const modeInfo = state.generationMode !== "single" ? ` (${state.generationMode} mode)` : "";
  sendLog("info", `Starting generation with ${state.prompts.length} prompts${modeInfo}`);
  await processNextPrompt();
}
function handleStopGeneration() {
  state.isRunning = false;
  state.isPaused = false;
  if (state.settings?.closeGrokTabsAfter && state.settings?.service === "grok") {
    closeGrokTabs();
  }
  state.activeSlots.clear();
  state.prompts = state.prompts.map(
    (p) => p.status === "processing" ? { ...p, status: "pending" } : p
  );
  sendLog("info", "Generation stopped");
  broadcastUpdate();
}
async function closeGrokTabs() {
  try {
    const tabs = await chrome.tabs.query({ url: "*://grok.com/*" });
    if (tabs.length > 1) {
      const toClose = tabs.slice(1).map((t) => t.id).filter((id) => id !== void 0);
      if (toClose.length > 0) {
        await chrome.tabs.remove(toClose);
        sendLog("info", `Closed ${toClose.length} Grok tab(s)`);
      }
    }
  } catch {
  }
}
function handleRetryFailed() {
  const failedPrompts = state.prompts.filter((p) => p.status === "failed");
  if (failedPrompts.length === 0) {
    sendLog("warning", "No failed prompts to retry");
    return;
  }
  state.prompts = state.prompts.map(
    (p) => p.status === "failed" ? { ...p, status: "pending", retryCount: p.retryCount + 1, error: void 0 } : p
  );
  sendLog("info", `Retrying ${failedPrompts.length} failed prompts`);
  if (!state.isRunning) {
    state.isRunning = true;
    processNextPrompt();
  }
}
async function processNextPrompt() {
  if (!state.isRunning || state.isPaused || !state.settings)
    return;
  const isFlowBased = ["veo3", "banana", "whisk"].includes(state.settings.service);
  const maxSlots = isFlowBased ? 1 : state.settings.maxVideoThreads;
  if (state.activeSlots.size >= maxSlots) {
    console.log(`[GenFlow] All slots busy (${state.activeSlots.size}/${maxSlots}), waiting...`);
    return;
  }
  const nextPrompt = state.prompts.find((p) => p.status === "pending");
  if (!nextPrompt) {
    if (state.activeSlots.size === 0) {
      state.isRunning = false;
      if (state.settings?.closeGrokTabsAfter && state.settings?.service === "grok") {
        closeGrokTabs();
      }
      sendLog("success", "All prompts processed");
      broadcastUpdate();
    }
    return;
  }
  console.log(`[GenFlow] Processing prompt #${nextPrompt.number}, slots: ${state.activeSlots.size}/${maxSlots}`);
  updatePromptStatus(nextPrompt.id, "processing");
  const slotId = Date.now();
  const tab = await getServiceTab(state.settings.service);
  if (!tab || !tab.id) {
    handleGenerationFailed({ promptId: nextPrompt.id, error: "Could not access service tab" });
    return;
  }
  state.activeSlots.set(slotId, { tabId: tab.id, promptId: nextPrompt.id });
  broadcastUpdate();
  try {
    await ensureContentScriptInjected(tab.id, state.settings.service);
    await new Promise((resolve) => setTimeout(resolve, 500));
    await chrome.tabs.sendMessage(tab.id, {
      type: "INJECT_PROMPT",
      payload: {
        prompt: nextPrompt,
        settings: state.settings,
        slotId,
        generationMode: state.generationMode
      }
    });
    sendLog("info", `Injecting prompt #${nextPrompt.number}`);
  } catch (error) {
    console.error("Failed to inject prompt:", error);
    handleGenerationFailed({
      promptId: nextPrompt.id,
      error: "Failed to inject prompt into page"
    });
    state.activeSlots.delete(slotId);
  }
}
function handleGenerationComplete(payload) {
  const prompt = state.prompts.find((p) => p.id === payload.promptId);
  if (!prompt)
    return;
  updatePromptStatus(payload.promptId, "completed", {
    resultUrl: payload.resultUrl,
    completedAt: Date.now()
  });
  for (const [slotId, slot] of state.activeSlots.entries()) {
    if (slot.promptId === payload.promptId) {
      state.activeSlots.delete(slotId);
      break;
    }
  }
  sendLog("success", `Prompt #${prompt.number} completed`);
  const service = state.settings?.service || "veo3";
  trackUsageOnServer(service);
  if (state.settings?.autoDownload && payload.resultUrl) {
    const filename = generateFilename(prompt);
    handleDownload({ url: payload.resultUrl, filename });
  }
  if (state.generationMode === "film" && payload.resultUrl) {
    state.lastResultUrl = payload.resultUrl;
    console.log(`[GenFlow] Film mode: saved result for chaining: ${payload.resultUrl.substring(0, 60)}...`);
    const nextPromptIndex = state.prompts.findIndex((p) => p.status === "pending");
    if (nextPromptIndex !== -1) {
      state.prompts[nextPromptIndex] = {
        ...state.prompts[nextPromptIndex],
        imageUrl: payload.resultUrl
      };
      console.log(`[GenFlow] Film mode: set imageUrl for prompt #${state.prompts[nextPromptIndex].number}`);
    }
  }
  const delay = state.settings ? getRandomDelay(state.settings) : 5e3;
  console.log(`[GenFlow] Prompt #${prompt.number} done. Next prompt in ${(delay / 1e3).toFixed(1)}s...`);
  setTimeout(() => {
    processNextPrompt();
  }, delay);
}
function handleGenerationFailed(payload) {
  const prompt = state.prompts.find((p) => p.id === payload.promptId);
  if (!prompt)
    return;
  const shouldRetry = state.settings?.autoRetryFailed && prompt.retryCount < (state.settings?.maxRetries || 1);
  if (shouldRetry) {
    updatePromptStatus(payload.promptId, "pending", {
      retryCount: prompt.retryCount + 1
    });
    sendLog("warning", `Prompt #${prompt.number} failed, retrying... (${prompt.retryCount + 1}/${state.settings?.maxRetries})`);
  } else {
    updatePromptStatus(payload.promptId, "failed", { error: payload.error });
    sendLog("error", `Prompt #${prompt.number} failed: ${payload.error}`);
  }
  for (const [slotId, slot] of state.activeSlots.entries()) {
    if (slot.promptId === payload.promptId) {
      state.activeSlots.delete(slotId);
      break;
    }
  }
  const delay = state.settings ? getRandomDelay(state.settings) : 5e3;
  console.log(`[GenFlow] Prompt #${prompt.number} failed. Next prompt in ${(delay / 1e3).toFixed(1)}s...`);
  setTimeout(() => {
    processNextPrompt();
  }, delay);
}
async function handleDownload(payload) {
  try {
    await chrome.downloads.download({
      url: payload.url,
      filename: payload.filename,
      saveAs: false
    });
    sendLog("info", `Downloaded: ${payload.filename}`);
  } catch (error) {
    sendLog("error", `Download failed: ${payload.filename}`);
  }
}
function generateFilename(prompt) {
  const prefix = state.settings?.filenamePrefix || "";
  const type = state.settings?.service === "veo3" || state.settings?.service === "grok" ? "Video" : "Photo";
  const hash = crypto.randomUUID().slice(0, 8);
  const ext = type === "Video" ? "mp4" : "png";
  return `${prefix}${prompt.number}_Prompt_${type}_${hash}.${ext}`;
}
function updatePromptStatus(id, status, updates) {
  state.prompts = state.prompts.map(
    (p) => p.id === id ? { ...p, status, ...updates } : p
  );
  broadcastUpdate();
}
var serviceTabCache = /* @__PURE__ */ new Map();
var originalActiveTabId = null;
function isExpectedServiceUrl(service, url) {
  if (!url)
    return false;
  const normalized = url.toLowerCase();
  const matchers = {
    veo3: /^https?:\/\/labs\.google(?:\.com)?\/fx\//,
    whisk: /^https?:\/\/labs\.google(?:\.com)?\/fx\/tools\/whisk\//,
    banana: /^https?:\/\/labs\.google(?:\.com)?\/fx\//,
    grok: /^https?:\/\/grok\.com\//
  };
  const pattern = matchers[service];
  return pattern ? pattern.test(normalized) : false;
}
async function saveCurrentTab() {
  try {
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (activeTab?.id) {
      originalActiveTabId = activeTab.id;
    }
  } catch {
  }
}
async function restoreFocusToOriginalTab() {
  if (originalActiveTabId) {
    try {
      await chrome.tabs.update(originalActiveTabId, { active: true });
    } catch {
    }
  }
}
async function getServiceTab(service) {
  await saveCurrentTab();
  const flowPatterns = ["*://labs.google/fx/*", "*://labs.google.com/fx/*"];
  const whiskPatterns = ["*://labs.google/fx/tools/whisk/*", "*://labs.google.com/fx/tools/whisk/*"];
  const urlPatterns = {
    veo3: [...flowPatterns],
    whisk: whiskPatterns,
    grok: ["*://grok.com/*"],
    banana: flowPatterns
  };
  const urls = {
    veo3: "https://labs.google/fx/ru/tools/flow",
    whisk: "https://labs.google/fx/tools/whisk/project",
    grok: "https://grok.com/imagine",
    banana: "https://labs.google/fx/ru/tools/flow"
  };
  const patterns = urlPatterns[service];
  if (!patterns)
    return null;
  if (service === "grok") {
    return await createNewServiceTab(service, urls[service]);
  }
  const cachedTabId = serviceTabCache.get(service);
  if (cachedTabId) {
    try {
      const tab = await chrome.tabs.get(cachedTabId);
      if (tab && tab.id && isExpectedServiceUrl(service, tab.url)) {
        console.log(`[GenFlow] Reusing cached tab ${cachedTabId} for ${service}`);
        await restoreFocusToOriginalTab();
        return tab;
      }
      serviceTabCache.delete(service);
    } catch {
      serviceTabCache.delete(service);
    }
  }
  for (const pattern of patterns) {
    const tabs = await chrome.tabs.query({ url: pattern });
    if (tabs.length > 0) {
      const tab = tabs[0];
      if (tab.id)
        serviceTabCache.set(service, tab.id);
      await restoreFocusToOriginalTab();
      return tab;
    }
  }
  const url = urls[service];
  if (!url)
    return null;
  return await createNewServiceTab(service, url);
}
async function createNewServiceTab(service, url) {
  sendLog("info", `Opening ${service} tab...`);
  const tab = await chrome.tabs.create({ url, active: false });
  if (tab.id && service !== "grok") {
    serviceTabCache.set(service, tab.id);
  }
  await restoreFocusToOriginalTab();
  await new Promise((resolve) => setTimeout(resolve, 5e3));
  await restoreFocusToOriginalTab();
  return tab;
}
function sendLog(type, message) {
  chrome.runtime.sendMessage({
    type: "LOG_MESSAGE",
    payload: { type, message }
  }).catch(() => {
  });
}
function broadcastUpdate() {
  chrome.runtime.sendMessage({
    type: "STATE_UPDATE",
    payload: {
      isRunning: state.isRunning,
      isPaused: state.isPaused,
      activeSlots: state.activeSlots.size,
      prompts: state.prompts
    }
  }).catch(() => {
  });
}
chrome.alarms.create("checkGenerations", { periodInMinutes: 0.5 });
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === "checkGenerations" && state.isRunning && !state.isPaused) {
    for (const [slotId, slot] of state.activeSlots.entries()) {
      if (Date.now() - slotId > 5 * 60 * 1e3) {
        sendLog("warning", `Slot ${slotId} appears stalled, releasing`);
        state.activeSlots.delete(slotId);
        const prompt = state.prompts.find((p) => p.id === slot.promptId);
        if (prompt && prompt.status === "processing") {
          handleGenerationFailed({
            promptId: slot.promptId,
            error: "Generation timed out"
          });
        }
      }
    }
  }
});
async function ensureContentScriptInjected(tabId, service) {
  const scriptMap = {
    veo3: "content/veo.js",
    whisk: "content/whisk.js",
    grok: "content/grok.js",
    banana: "content/banana.js"
  };
  const scriptFile = scriptMap[service];
  if (!scriptFile)
    return;
  try {
    await chrome.tabs.sendMessage(tabId, { type: "PING" });
    console.log(`[GenFlow] Content script already loaded for ${service}`);
  } catch {
    console.log(`[GenFlow] Injecting content script for ${service}`);
    await chrome.scripting.executeScript({
      target: { tabId },
      files: [scriptFile]
    });
  }
}
chrome.action.onClicked.addListener(async (tab) => {
  if (tab.windowId) {
    await chrome.sidePanel.open({ windowId: tab.windowId });
  }
});
chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true }).catch(() => {
});
console.log("GenFlow background service worker initialized");
