"use strict";
(() => {
  // src/content/grok.ts
  var currentPromptId = null;
  var monitorInterval = null;
  console.log("[GenFlow] Grok content script loaded:", window.location.href);
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    console.log("[GenFlow] Grok received message:", message.type);
    if (message.type === "PING") {
      sendResponse({ pong: true });
      return false;
    }
    if (message.type === "INJECT_PROMPT") {
      const payload = message.payload;
      if (payload.settings.generationType !== "image-to-video") {
        chrome.runtime.sendMessage({
          type: "GENERATION_FAILED",
          payload: { promptId: payload.prompt.id, error: "Grok supports only Image to Video mode" }
        });
        sendResponse({ success: false });
        return false;
      }
      if (!payload.prompt.imageUrl) {
        chrome.runtime.sendMessage({
          type: "GENERATION_FAILED",
          payload: { promptId: payload.prompt.id, error: "Image is required for Grok" }
        });
        sendResponse({ success: false });
        return false;
      }
      injectPrompt(payload);
      sendResponse({ success: true });
      return false;
    }
    return false;
  });
  function sleep(ms) {
    return new Promise((r) => setTimeout(r, ms));
  }
  function isVisible(el) {
    if (!el)
      return false;
    const style = window.getComputedStyle(el);
    return style.display !== "none" && style.visibility !== "hidden" && style.opacity !== "0" && el.offsetWidth > 0 && el.offsetHeight > 0;
  }
  function isOnImaginePage() {
    return window.location.pathname.startsWith("/imagine") && !window.location.pathname.includes("/imagine/post/");
  }
  function isOnResultPage() {
    return window.location.pathname.includes("/imagine/post/");
  }
  async function ensureOnImaginePage() {
    if (isOnImaginePage()) {
      console.log("[GenFlow] Grok: Already on /imagine");
      return;
    }
    if (isOnResultPage()) {
      console.log("[GenFlow] Grok: On result page, navigating to /imagine...");
      window.location.href = "https://grok.com/imagine";
      await waitForPageLoad();
      return;
    }
    console.log("[GenFlow] Grok: Not on /imagine, navigating...");
    window.location.href = "https://grok.com/imagine";
    await waitForPageLoad();
  }
  async function waitForPageLoad() {
    for (let i = 0; i < 30; i++) {
      await sleep(500);
      if (isOnImaginePage()) {
        await sleep(1500);
        return;
      }
    }
    throw new Error("Failed to navigate to /imagine");
  }
  function findImagineInput() {
    const textareas = document.querySelectorAll("textarea");
    for (const ta of textareas) {
      if (isVisible(ta)) {
        const placeholder = ta.getAttribute("placeholder") || "";
        console.log(`[GenFlow] Grok: Found textarea with placeholder: "${placeholder.substring(0, 50)}"`);
        return ta;
      }
    }
    const pm = document.querySelector(".tiptap.ProseMirror, .ProseMirror");
    if (pm && isVisible(pm)) {
      console.log("[GenFlow] Grok: Found ProseMirror input");
      return pm;
    }
    const ce = document.querySelector('[contenteditable="true"]');
    if (ce && isVisible(ce)) {
      console.log("[GenFlow] Grok: Found contenteditable input");
      return ce;
    }
    return null;
  }
  async function uploadImageToImagine(imageUrl) {
    console.log("[GenFlow] Grok: Uploading image...");
    let fileInput = document.querySelector('input[type="file"][accept*="image"]');
    if (!fileInput) {
      fileInput = document.querySelector('input[type="file"]');
    }
    if (!fileInput) {
      const uploadBtns = document.querySelectorAll("button");
      for (const btn of uploadBtns) {
        const label = btn.getAttribute("aria-label") || "";
        const text = (btn.textContent || "").toLowerCase();
        if ((label.includes("upload") || label.includes("\u0437\u0430\u0433\u0440\u0443\u0437\u0438\u0442\u044C") || label.includes("attach") || label.includes("\u043F\u0440\u0438\u043A\u0440\u0435\u043F\u0438\u0442\u044C") || text.includes("upload") || text.includes("\u0437\u0430\u0433\u0440\u0443\u0437\u0438\u0442\u044C")) && isVisible(btn)) {
          console.log(`[GenFlow] Grok: Clicking upload button: "${label || text}"`);
          btn.click();
          await sleep(1e3);
          break;
        }
      }
      fileInput = document.querySelector('input[type="file"]');
    }
    if (!fileInput) {
      throw new Error("File input not found on /imagine");
    }
    const response = await fetch(imageUrl);
    const blob = await response.blob();
    const file = new File([blob], `grok_image_${Date.now()}.png`, { type: "image/png" });
    const dt = new DataTransfer();
    dt.items.add(file);
    fileInput.files = dt.files;
    fileInput.dispatchEvent(new Event("change", { bubbles: true }));
    fileInput.dispatchEvent(new Event("input", { bubbles: true }));
    console.log("[GenFlow] Grok: Image uploaded");
    await sleep(2e3);
  }
  function enterPromptText(inputEl, text) {
    const promptText = `Create a video: ${text}`;
    if (inputEl.tagName === "TEXTAREA") {
      const ta = inputEl;
      const setter = Object.getOwnPropertyDescriptor(HTMLTextAreaElement.prototype, "value")?.set;
      if (setter) {
        setter.call(ta, promptText);
      } else {
        ta.value = promptText;
      }
      ta.dispatchEvent(new Event("input", { bubbles: true }));
      ta.dispatchEvent(new Event("change", { bubbles: true }));
    } else if (inputEl.getAttribute("contenteditable") === "true") {
      inputEl.innerHTML = "";
      const p = document.createElement("p");
      p.textContent = promptText;
      inputEl.appendChild(p);
      inputEl.dispatchEvent(new InputEvent("input", { bubbles: true, inputType: "insertText" }));
    } else {
      inputEl.value = promptText;
      inputEl.dispatchEvent(new Event("input", { bubbles: true }));
    }
    inputEl.focus();
  }
  function findSendButton() {
    const labels = [
      "\u0421\u0434\u0435\u043B\u0430\u0442\u044C \u0432\u0438\u0434\u0435\u043E",
      "Make video",
      "\u0421\u043E\u0437\u0434\u0430\u0442\u044C \u0432\u0438\u0434\u0435\u043E",
      "Generate",
      "Submit",
      "Send",
      "\u041E\u0442\u043F\u0440\u0430\u0432\u0438\u0442\u044C"
    ];
    for (const label of labels) {
      const btn = document.querySelector(`button[aria-label="${label}"]`);
      if (btn && isVisible(btn) && !btn.disabled) {
        return btn;
      }
    }
    const allBtns = document.querySelectorAll("button");
    for (const btn of allBtns) {
      const text = (btn.textContent || "").trim();
      if (labels.some((l) => text.includes(l)) && isVisible(btn) && !btn.disabled) {
        return btn;
      }
    }
    const submit = document.querySelector('button[type="submit"]');
    if (submit && isVisible(submit) && !submit.disabled)
      return submit;
    return null;
  }
  async function injectPrompt(payload) {
    const { prompt, settings, slotId } = payload;
    currentPromptId = prompt.id;
    if (monitorInterval) {
      clearInterval(monitorInterval);
      monitorInterval = null;
    }
    try {
      console.log(`[GenFlow] Grok: Starting prompt #${prompt.number}`);
      await ensureOnImaginePage();
      await sleep(1500);
      await uploadImageToImagine(prompt.imageUrl);
      const inputEl = findImagineInput();
      if (!inputEl) {
        throw new Error("Prompt input not found on /imagine");
      }
      enterPromptText(inputEl, prompt.text);
      await sleep(500);
      let sendBtn = null;
      for (let i = 0; i < 10; i++) {
        sendBtn = findSendButton();
        if (sendBtn)
          break;
        await sleep(500);
      }
      if (!sendBtn) {
        throw new Error("Send/Generate button not found");
      }
      const urlBeforeClick = window.location.href;
      sendBtn.click();
      console.log(`[GenFlow] Grok: Generation started for prompt #${prompt.number}`);
      startCompletionMonitor(prompt, slotId, urlBeforeClick);
    } catch (error) {
      console.error("[GenFlow] Grok injection failed:", error);
      chrome.runtime.sendMessage({
        type: "GENERATION_FAILED",
        payload: { promptId: prompt.id, error: error instanceof Error ? error.message : "Injection failed" }
      });
    }
  }
  function startCompletionMonitor(prompt, slotId, urlBeforeClick) {
    const startTime = Date.now();
    const timeout = 10 * 60 * 1e3;
    let checkCount = 0;
    let lastVideoSrc = "";
    console.log("[GenFlow] Grok: Starting completion monitor");
    monitorInterval = setInterval(async () => {
      checkCount++;
      if (currentPromptId !== prompt.id) {
        clearInterval(monitorInterval);
        monitorInterval = null;
        return;
      }
      const progressEls = document.querySelectorAll('.tabular-nums, [class*="progress"]');
      for (const el of progressEls) {
        const text = (el.textContent || "").trim();
        const match = text.match(/(\d+)%/);
        if (match) {
          const pct = parseInt(match[1]);
          if (pct > 0 && pct < 100) {
            if (checkCount % 5 === 0) {
              console.log(`[GenFlow] Grok: Progress ${pct}%`);
            }
            return;
          }
        }
      }
      const videos = document.querySelectorAll("video");
      for (const video of videos) {
        const src = video.getAttribute("src") || video.currentSrc || "";
        if (src && src.includes(".mp4") && src !== lastVideoSrc && isVisible(video)) {
          const rect = video.getBoundingClientRect();
          if (rect.width > 100 && rect.height > 50) {
            lastVideoSrc = src;
            const dlBtn = document.querySelector('button[aria-label*="\u043A\u0430\u0447\u0430\u0442\u044C"], button[aria-label*="ownload"], a[download]');
            if (dlBtn || checkCount > 5) {
              clearInterval(monitorInterval);
              monitorInterval = null;
              console.log(`[GenFlow] Grok: Video complete for prompt #${prompt.number}: ${src.substring(0, 80)}`);
              if (src.startsWith("http")) {
                chrome.runtime.sendMessage({
                  type: "DOWNLOAD_RESULT",
                  payload: { url: src, filename: `Grok_${prompt.number}_${Date.now()}.mp4` }
                });
              }
              chrome.runtime.sendMessage({
                type: "GENERATION_COMPLETE",
                payload: { promptId: prompt.id, resultUrl: src }
              });
              return;
            }
          }
        }
      }
      const retryBtns = document.querySelectorAll("button");
      for (const btn of retryBtns) {
        const text = (btn.textContent || "").trim();
        if ((text.includes("\u041F\u043E\u0432\u0442\u043E\u0440\u0438\u0442\u044C") || text.includes("Retry")) && isVisible(btn)) {
          if (lastVideoSrc) {
            clearInterval(monitorInterval);
            monitorInterval = null;
            chrome.runtime.sendMessage({
              type: "GENERATION_COMPLETE",
              payload: { promptId: prompt.id, resultUrl: lastVideoSrc }
            });
            return;
          }
        }
      }
      if (checkCount % 10 === 0) {
        console.log(`[GenFlow] Grok: Check #${checkCount}, waiting...`);
      }
      if (Date.now() - startTime > timeout) {
        clearInterval(monitorInterval);
        monitorInterval = null;
        chrome.runtime.sendMessage({
          type: "GENERATION_FAILED",
          payload: { promptId: prompt.id, error: "Generation timed out (10 min)" }
        });
      }
    }, 3e3);
  }
  console.log("[GenFlow] Grok Imagine script ready");
})();
