"use strict";
(() => {
  // src/content/veo.ts
  var currentPromptId = null;
  var generationCheckInterval = null;
  var FLOW_SELECTORS = {
    // Main prompt textarea - has unique ID
    promptTextarea: "#PINHOLE_TEXT_AREA_ELEMENT_ID",
    promptTextareaFallbacks: [
      'textarea[placeholder*="\u0421\u0433\u0435\u043D\u0435\u0440\u0438\u0440\u0443\u0439\u0442\u0435"]',
      'textarea[placeholder*="\u0442\u0435\u043A\u0441\u0442\u043E\u0432\u043E\u043C\u0443 \u0437\u0430\u043F\u0440\u043E\u0441\u0443"]',
      'textarea[placeholder*="Generate"]',
      'textarea[placeholder*="text prompt"]',
      "textarea.sc-e586993-0",
      "textarea.gEBbLp",
      "textarea"
    ],
    // Create button (arrow_forward icon + "Создать")
    // From diagnostic: class="sc-c177465c-1 gdArnN sc-408537d4-2 gdXWm"
    createButton: "button.gdXWm, button.gdArnN, button.sc-408537d4-2",
    createButtonOverlay: 'button.gdArnN div[data-type="button-overlay"]',
    // Create project button on list page (add_2 icon)
    createProjectButton: "button.sc-a38764c7-0, button.fXsrxE",
    // Mode dropdown button (Видео по описанию / arrow_drop_down)
    // From diagnostic: class="sc-c177465c-1 hVamcH sc-bd77098e-1 hdOBaY sc-fbe1c021-0 hKBF"
    modeDropdown: "button.hdOBaY, button.hKBFUo, button.sc-bd77098e-1",
    // Model selector button (Veo 3.1 - Fast)
    modelButton: "button.sc-fd871d0e-4, button.flUTPU",
    // Settings button (tune icon)
    settingsButton: "button.sc-3d54f340-0, button.fFoTDV",
    // Tab buttons (Video / Images)
    tabVideo: 'button:has(span:contains("\u0412\u0438\u0434\u0435\u043E")), button:has(span:contains("Video"))',
    tabImages: 'button:has(span:contains("\u0418\u0437\u043E\u0431\u0440\u0430\u0436\u0435\u043D\u0438\u044F")), button:has(span:contains("Images"))',
    // Generated content indicators
    videoPreview: "video[src], video source[src]",
    downloadButton: 'a[download], button[aria-label*="download" i], button[aria-label*="\u0441\u043A\u0430\u0447\u0430\u0442\u044C" i]',
    generatingSpinner: '.generating, .loading, [class*="spinner"], [class*="loading"]',
    // Error indicators
    errorMessage: '[role="alert"], .error, [class*="error"]'
  };
  var FLOW_TEXTS = {
    createProject: ["\u0421\u043E\u0437\u0434\u0430\u0442\u044C \u043F\u0440\u043E\u0435\u043A\u0442", "Create project", "Create Project", "New project"],
    createButton: ["\u0421\u043E\u0437\u0434\u0430\u0442\u044C", "Create", "Generate"],
    videoTab: ["\u0412\u0438\u0434\u0435\u043E", "Video"],
    imageTab: ["\u0418\u0437\u043E\u0431\u0440\u0430\u0436\u0435\u043D\u0438\u044F", "Images"],
    modes: {
      "text-to-video": ["\u0412\u0438\u0434\u0435\u043E \u043F\u043E \u043E\u043F\u0438\u0441\u0430\u043D\u0438\u044E", "Text to Video", "Video from text", "Text-to-Video"],
      // "Видео по кадрам" is the correct option for image-to-video on Flow
      "image-to-video": ["\u0412\u0438\u0434\u0435\u043E \u043F\u043E \u043A\u0430\u0434\u0440\u0430\u043C", "Frames to Video", "Video from frames", "Image to Video", "Frames-to-Video", "\u0412\u0438\u0434\u0435\u043E \u0438\u0437 \u043A\u0430\u0434\u0440\u043E\u0432"]
    }
  };
  chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
    if (message.type === "PING") {
      sendResponse({ pong: true });
      return false;
    }
    if (message.type === "INJECT_PROMPT") {
      const payload = message.payload;
      if (isFlowHost() && payload.settings?.service !== "veo3")
        return false;
      injectPrompt(payload);
      sendResponse({ success: true });
      return false;
    }
    if (message.type === "CHECK_GENERATION_STATUS") {
      const status = checkGenerationStatus();
      sendResponse(status);
      return false;
    }
    return false;
  });
  function isFlowHost() {
    return /labs\.google(\.com)?\/fx/i.test(window.location.href);
  }
  function isVisible(el) {
    const style = window.getComputedStyle(el);
    return style.display !== "none" && style.visibility !== "hidden" && style.opacity !== "0" && el.offsetWidth > 0 && el.offsetHeight > 0;
  }
  function findElementByText(texts, tagName = "button") {
    const elements = document.querySelectorAll(tagName);
    for (const el of elements) {
      const text = (el.textContent || "").trim();
      if (texts.some((t) => text.includes(t))) {
        return el;
      }
    }
    return null;
  }
  function findPromptTextarea() {
    const byId = document.querySelector(FLOW_SELECTORS.promptTextarea);
    if (byId && isVisible(byId)) {
      console.log("[GenFlow] Found prompt textarea by ID");
      return byId;
    }
    for (const sel of FLOW_SELECTORS.promptTextareaFallbacks) {
      const el = document.querySelector(sel);
      if (el && isVisible(el)) {
        console.log(`[GenFlow] Found prompt textarea with: ${sel}`);
        return el;
      }
    }
    return null;
  }
  function findCreateButton() {
    const selectors = FLOW_SELECTORS.createButton.split(", ");
    for (const sel of selectors) {
      const el = document.querySelector(sel);
      if (el && !el.disabled && isVisible(el)) {
        console.log(`[GenFlow] Found create button with: ${sel}`);
        return el;
      }
    }
    const byText = findElementByText(FLOW_TEXTS.createButton);
    if (byText && !byText.disabled) {
      console.log("[GenFlow] Found create button by text");
      return byText;
    }
    const overlay = document.querySelector(FLOW_SELECTORS.createButtonOverlay);
    if (overlay) {
      const btn = overlay.closest("button");
      if (btn && !btn.disabled) {
        console.log("[GenFlow] Found create button via overlay");
        return btn;
      }
    }
    return null;
  }
  function findCreateProjectButton() {
    const selectors = FLOW_SELECTORS.createProjectButton.split(", ");
    for (const sel of selectors) {
      const el = document.querySelector(sel);
      if (el && isVisible(el)) {
        const text = (el.textContent || "").trim();
        if (FLOW_TEXTS.createProject.some((t) => text.includes(t))) {
          return el;
        }
      }
    }
    return findElementByText(FLOW_TEXTS.createProject);
  }
  function isOnFlowProjectPage() {
    return /\/project\/[a-f0-9-]+/i.test(window.location.pathname);
  }
  async function ensureFlowProjectPage() {
    if (isOnFlowProjectPage()) {
      console.log("[GenFlow] Already on project page, reusing...");
      await waitForPageReady();
      return;
    }
    console.log("[GenFlow] Not on project page, navigating...");
    await sleep(2e3);
    const existingProjects = document.querySelectorAll('a[href*="/project/"], [href*="/project/"]');
    if (existingProjects.length > 0) {
      console.log("[GenFlow] Found existing project, opening...");
      existingProjects[0].click();
      for (let i = 0; i < 30; i++) {
        await sleep(500);
        if (isOnFlowProjectPage())
          break;
      }
      if (isOnFlowProjectPage()) {
        await waitForPageReady();
        return;
      }
    }
    let createProjectBtn = null;
    for (let i = 0; i < 40; i++) {
      createProjectBtn = findCreateProjectButton();
      if (createProjectBtn)
        break;
      await sleep(500);
    }
    if (!createProjectBtn) {
      throw new Error("\u041A\u043D\u043E\u043F\u043A\u0430 \xAB\u0421\u043E\u0437\u0434\u0430\u0442\u044C \u043F\u0440\u043E\u0435\u043A\u0442\xBB \u043D\u0435 \u043D\u0430\u0439\u0434\u0435\u043D\u0430. \u041E\u0442\u043A\u0440\u043E\u0439\u0442\u0435 https://labs.google/fx/ru/tools/flow \u0438 \u043D\u0430\u0436\u043C\u0438\u0442\u0435 \u0435\u0451 \u0432\u0440\u0443\u0447\u043D\u0443\u044E.");
    }
    console.log("[GenFlow] Clicking create project button...");
    createProjectBtn.click();
    for (let i = 0; i < 50; i++) {
      await sleep(500);
      if (isOnFlowProjectPage()) {
        console.log("[GenFlow] Redirected to project page");
        break;
      }
    }
    if (!isOnFlowProjectPage()) {
      throw new Error("\u041D\u0435 \u0443\u0434\u0430\u043B\u043E\u0441\u044C \u043F\u0435\u0440\u0435\u0439\u0442\u0438 \u043D\u0430 \u0441\u0442\u0440\u0430\u043D\u0438\u0446\u0443 \u043F\u0440\u043E\u0435\u043A\u0442\u0430. \u041D\u0430\u0436\u043C\u0438\u0442\u0435 \xAB\u0421\u043E\u0437\u0434\u0430\u0442\u044C \u043F\u0440\u043E\u0435\u043A\u0442\xBB \u0432\u0440\u0443\u0447\u043D\u0443\u044E \u0438 \u043F\u043E\u0432\u0442\u043E\u0440\u0438\u0442\u0435.");
    }
    await waitForPageReady();
  }
  async function waitForPageReady() {
    console.log("[GenFlow] Waiting for page to be ready...");
    for (let i = 0; i < 30; i++) {
      const textarea = findPromptTextarea();
      if (textarea) {
        console.log("[GenFlow] Page is ready");
        await sleep(500);
        return;
      }
      await sleep(500);
    }
    throw new Error("\u0421\u0442\u0440\u0430\u043D\u0438\u0446\u0430 \u043F\u0440\u043E\u0435\u043A\u0442\u0430 \u043D\u0435 \u0437\u0430\u0433\u0440\u0443\u0437\u0438\u043B\u0430\u0441\u044C. \u041F\u043E\u043F\u0440\u043E\u0431\u0443\u0439\u0442\u0435 \u043E\u0431\u043D\u043E\u0432\u0438\u0442\u044C \u0441\u0442\u0440\u0430\u043D\u0438\u0446\u0443.");
  }
  async function selectFlowMode(generationType) {
    console.log(`[GenFlow] Selecting mode: ${generationType}`);
    const targetTexts = FLOW_TEXTS.modes[generationType] || [];
    console.log(`[GenFlow] Target texts: ${targetTexts.join(", ")}`);
    let dropdownBtn = null;
    const allButtons = document.querySelectorAll("button");
    for (const btn of allButtons) {
      const text = (btn.textContent || "").trim();
      if ((text.includes("\u0412\u0438\u0434\u0435\u043E \u043F\u043E") || text.includes("\u0421\u043E\u0437\u0434\u0430\u0442\u044C") || text.includes("Video")) && isVisible(btn)) {
        const hasArrow = btn.querySelector("svg") || text.includes("arrow") || btn.querySelector('[class*="chevron"]');
        if (hasArrow) {
          dropdownBtn = btn;
          console.log(`[GenFlow] Found dropdown button: "${text}"`);
          break;
        }
      }
    }
    if (!dropdownBtn) {
      const selectors = FLOW_SELECTORS.modeDropdown.split(", ");
      for (const sel of selectors) {
        const el = document.querySelector(sel);
        if (el && isVisible(el)) {
          dropdownBtn = el;
          console.log(`[GenFlow] Found dropdown by selector: ${sel}`);
          break;
        }
      }
    }
    if (!dropdownBtn) {
      console.log("[GenFlow] Mode dropdown not found, continuing with default mode");
      return;
    }
    const currentText = (dropdownBtn.textContent || "").toLowerCase();
    if (targetTexts.some((t) => currentText.includes(t.toLowerCase()))) {
      console.log(`[GenFlow] Already in correct mode: "${currentText}"`);
      return;
    }
    console.log(`[GenFlow] Current mode: "${currentText}", need to switch to: ${generationType}`);
    dropdownBtn.click();
    await sleep(800);
    console.log("[GenFlow] Looking for dropdown options...");
    const logAllOptions = () => {
      const allClickable = document.querySelectorAll('button, [role="option"], [role="menuitem"], div[class*="option"], div[class*="item"]');
      console.log(`[GenFlow] Found ${allClickable.length} clickable elements`);
      allClickable.forEach((el, i) => {
        const text = (el.textContent || "").trim().substring(0, 50);
        if (text && isVisible(el)) {
          console.log(`[GenFlow] Option ${i}: "${text}"`);
        }
      });
    };
    logAllOptions();
    const findAndClickOption = () => {
      const exactTargets = generationType === "image-to-video" ? ["\u0412\u0438\u0434\u0435\u043E \u043F\u043E \u043A\u0430\u0434\u0440\u0430\u043C", "\u0432\u0438\u0434\u0435\u043E \u043F\u043E \u043A\u0430\u0434\u0440\u0430\u043C"] : ["\u0412\u0438\u0434\u0435\u043E \u043F\u043E \u043E\u043F\u0438\u0441\u0430\u043D\u0438\u044E", "\u0432\u0438\u0434\u0435\u043E \u043F\u043E \u043E\u043F\u0438\u0441\u0430\u043D\u0438\u044E"];
      const allElements = document.querySelectorAll("*");
      for (const el of allElements) {
        const directText = Array.from(el.childNodes).filter((n) => n.nodeType === Node.TEXT_NODE).map((n) => n.textContent?.trim()).join("");
        const fullText = (el.textContent || "").trim();
        for (const target of exactTargets) {
          if (directText.includes(target) || fullText === target) {
            if (isVisible(el) && el.offsetHeight > 0) {
              console.log(`[GenFlow] Found exact match: "${fullText}", clicking...`);
              el.click();
              return true;
            }
          }
        }
      }
      const roleOptions = document.querySelectorAll('[role="option"], [role="menuitem"], [role="menuitemradio"]');
      for (const option of roleOptions) {
        const text = (option.textContent || "").trim();
        console.log(`[GenFlow] Role option: "${text}"`);
        if (targetTexts.some((t) => text.toLowerCase().includes(t.toLowerCase()))) {
          console.log(`[GenFlow] Clicking role option: ${text}`);
          option.click();
          return true;
        }
      }
      const radixOptions = document.querySelectorAll("[data-radix-collection-item]");
      for (const option of radixOptions) {
        const text = (option.textContent || "").trim();
        if (targetTexts.some((t) => text.toLowerCase().includes(t.toLowerCase()))) {
          console.log(`[GenFlow] Clicking radix option: ${text}`);
          option.click();
          return true;
        }
      }
      const popovers = document.querySelectorAll('[data-state="open"], [role="listbox"], [role="menu"], [class*="popover"], [class*="dropdown"], [class*="menu"]');
      for (const popover of popovers) {
        const items = popover.querySelectorAll("div, button, li, span");
        for (const el of items) {
          const text = (el.textContent || "").trim();
          if (targetTexts.some((t) => text.toLowerCase().includes(t.toLowerCase()))) {
            if (isVisible(el)) {
              console.log(`[GenFlow] Clicking popover item: ${text}`);
              el.click();
              return true;
            }
          }
        }
      }
      const clickables = document.querySelectorAll('button, div[role="button"], [class*="option"], [class*="item"], [class*="menu"] > *');
      for (const btn of clickables) {
        const text = (btn.textContent || "").trim();
        if (targetTexts.some((t) => text.toLowerCase().includes(t.toLowerCase()))) {
          if (isVisible(btn)) {
            console.log(`[GenFlow] Clicking by text match: ${text}`);
            btn.click();
            return true;
          }
        }
      }
      return false;
    };
    if (findAndClickOption()) {
      await sleep(500);
      return;
    }
    closeFlowDialogIfOpen();
    console.log("[GenFlow] Target mode not found in dropdown. Available texts were logged above.");
  }
  async function uploadImageForVideo(imageUrl, endImageUrl) {
    console.log("[GenFlow] Uploading image(s) for video generation...");
    console.log(`[GenFlow] Start image: ${imageUrl ? "yes" : "no"}, End image: ${endImageUrl ? "yes" : "no"}`);
    const findImageSlotButtons = () => {
      const uploadArea = document.querySelector('.sc-aa137585-0, .jVsOOk, [class*="upload"]');
      let allAddButtons = [];
      const specificButtons = document.querySelectorAll("button.sc-d02e9a37-1");
      specificButtons.forEach((btn) => {
        if (isVisible(btn)) {
          allAddButtons.push(btn);
        }
      });
      if (allAddButtons.length < 2) {
        const allButtons = document.querySelectorAll("button");
        allButtons.forEach((btn) => {
          const text = (btn.textContent || "").trim().toLowerCase();
          const hasAddIcon = btn.querySelector("i")?.textContent?.includes("add");
          if ((text === "add" || hasAddIcon) && isVisible(btn)) {
            if (!allAddButtons.includes(btn)) {
              allAddButtons.push(btn);
            }
          }
        });
      }
      allAddButtons.sort((a, b) => {
        const rectA = a.getBoundingClientRect();
        const rectB = b.getBoundingClientRect();
        return rectA.left - rectB.left;
      });
      console.log(`[GenFlow] Found ${allAddButtons.length} add buttons`);
      allAddButtons.forEach((btn, i) => {
        const rect = btn.getBoundingClientRect();
        console.log(`  Button ${i}: pos=(${Math.round(rect.left)}, ${Math.round(rect.top)}) class=${btn.className.substring(0, 50)}`);
      });
      return {
        start: allAddButtons[0] || null,
        end: allAddButtons[1] || null
      };
    };
    const uploadToFileInput = async (dataUrl, name) => {
      const fileInput = document.querySelector('input[type="file"][accept*=".png"], input[type="file"][accept*="image"]');
      if (!fileInput) {
        console.log("[GenFlow] No file input found");
        return false;
      }
      try {
        console.log(`[GenFlow] Fetching ${name} image...`);
        const response = await fetch(dataUrl);
        const blob = await response.blob();
        const file = new File([blob], `${name}_${Date.now()}.png`, { type: "image/png" });
        const dataTransfer = new DataTransfer();
        dataTransfer.items.add(file);
        fileInput.files = dataTransfer.files;
        fileInput.dispatchEvent(new Event("change", { bubbles: true }));
        fileInput.dispatchEvent(new Event("input", { bubbles: true }));
        console.log(`[GenFlow] ${name} image uploaded successfully`);
        return true;
      } catch (error) {
        console.error(`[GenFlow] ${name} upload error:`, error);
        return false;
      }
    };
    const buttons = findImageSlotButtons();
    console.log("[GenFlow] Step 1: Uploading START image...");
    if (buttons.start) {
      console.log("[GenFlow] Clicking START button...");
      buttons.start.click();
      await sleep(800);
      if (await uploadToFileInput(imageUrl, "start")) {
        await sleep(2e3);
      }
    } else {
      console.log("[GenFlow] No START button found, trying direct file input...");
      await uploadToFileInput(imageUrl, "start");
      await sleep(2e3);
    }
    if (endImageUrl) {
      console.log("[GenFlow] Step 2: Uploading END image...");
      const buttonsAfterFirst = findImageSlotButtons();
      if (buttonsAfterFirst.end) {
        console.log("[GenFlow] Clicking END button...");
        buttonsAfterFirst.end.click();
        await sleep(800);
        if (await uploadToFileInput(endImageUrl, "end")) {
          await sleep(2e3);
        }
      } else if (buttonsAfterFirst.start) {
        console.log("[GenFlow] Only one button found, using it for END image...");
        buttonsAfterFirst.start.click();
        await sleep(800);
        if (await uploadToFileInput(endImageUrl, "end")) {
          await sleep(2e3);
        }
      } else {
        console.log("[GenFlow] No END button found, trying direct file input...");
        await uploadToFileInput(endImageUrl, "end");
        await sleep(1500);
      }
    }
    console.log("[GenFlow] Image upload process completed");
  }
  function checkGenerationStatus() {
    const status = {
      isGenerating: false,
      isComplete: false,
      hasError: false
    };
    const spinners = document.querySelectorAll('[class*="loading"], [class*="spinner"], [class*="generating"], [class*="progress"], [class*="Loading"], [class*="Spinner"]');
    for (const spinner of spinners) {
      if (isVisible(spinner)) {
        const rect = spinner.getBoundingClientRect();
        if (rect.width > 20 && rect.height > 20) {
          status.isGenerating = true;
          break;
        }
      }
    }
    const videos = document.querySelectorAll("video");
    for (const video of videos) {
      const videoEl = video;
      const src = videoEl.src || videoEl.currentSrc || "";
      const posterSrc = videoEl.poster || "";
      if (src && (src.startsWith("http") || src.startsWith("blob:")) || posterSrc && posterSrc.startsWith("http")) {
        if (isVisible(videoEl)) {
          const rect = videoEl.getBoundingClientRect();
          if (rect.width > 100 && rect.height > 50) {
            status.isComplete = true;
            status.videoUrl = src || posterSrc;
            console.log("[GenFlow] Found video element:", src || posterSrc);
            break;
          }
        }
      }
      const sources = video.querySelectorAll("source");
      for (const source of sources) {
        const srcVal = source.getAttribute("src") || "";
        if (srcVal && (srcVal.startsWith("http") || srcVal.startsWith("blob:"))) {
          status.isComplete = true;
          status.videoUrl = srcVal;
          console.log("[GenFlow] Found video source:", srcVal);
          break;
        }
      }
    }
    if (!status.isComplete) {
      const playButtons = document.querySelectorAll('button svg path[d*="M8"], [class*="play"], [aria-label*="play" i], [aria-label*="\u0432\u043E\u0441\u043F\u0440\u043E\u0438\u0437\u0432" i]');
      for (const playBtn of playButtons) {
        const container = playBtn.closest("div");
        if (container) {
          const nearbyVideo = container.querySelector("video");
          const nearbyImg = container.querySelector('img[src*="http"]');
          if (nearbyVideo || nearbyImg) {
            status.isComplete = true;
            status.videoUrl = nearbyVideo?.src || nearbyImg?.src || window.location.href;
            console.log("[GenFlow] Found video via play button");
            break;
          }
        }
      }
    }
    if (!status.isComplete) {
      const veoLabels = document.querySelectorAll("span, div");
      for (const label of veoLabels) {
        const text = (label.textContent || "").trim();
        if (text === "Veo" || text === "\xA9 Veo") {
          const parent = label.closest('div[class*="video"], div[class*="player"], div[class*="preview"]');
          if (parent && isVisible(parent)) {
            status.isComplete = true;
            status.videoUrl = window.location.href;
            console.log("[GenFlow] Found Veo watermark, generation complete");
            break;
          }
        }
      }
    }
    const downloadBtns = document.querySelectorAll('a[download], [aria-label*="download" i], [aria-label*="\u0441\u043A\u0430\u0447\u0430\u0442\u044C" i], button[aria-label*="\u0421\u043A\u0430\u0447\u0430\u0442\u044C"], button[aria-label*="Download"]');
    for (const btn of downloadBtns) {
      if (isVisible(btn)) {
        status.isComplete = true;
        const href = btn.getAttribute("href");
        if (href)
          status.videoUrl = href;
        console.log("[GenFlow] Found download button");
        break;
      }
    }
    if (!status.isComplete) {
      const thumbnails = document.querySelectorAll('img[src*="googleusercontent"], img[src*="storage.googleapis"], img[src*="lh3.google"]');
      for (const thumb of thumbnails) {
        if (isVisible(thumb)) {
          const rect = thumb.getBoundingClientRect();
          if (rect.width > 150 && rect.height > 80) {
            status.isComplete = true;
            status.videoUrl = thumb.src;
            console.log("[GenFlow] Found video thumbnail:", status.videoUrl);
            break;
          }
        }
      }
    }
    if (!status.isComplete && !status.isGenerating) {
      const mediaContainers = document.querySelectorAll('[class*="video"], [class*="media"], [class*="player"], [class*="preview"]');
      for (const container of mediaContainers) {
        if (isVisible(container)) {
          const rect = container.getBoundingClientRect();
          if (rect.width > 200 && rect.height > 100) {
            const hasVideo = container.querySelector("video");
            const hasCanvas = container.querySelector("canvas");
            const hasImg = container.querySelector('img[src*="http"]');
            if (hasVideo || hasCanvas || hasImg) {
              status.isComplete = true;
              status.videoUrl = hasVideo?.src || hasImg?.src || window.location.href;
              console.log("[GenFlow] Found media container with content");
              break;
            }
          }
        }
      }
    }
    const POLICY_KEYWORDS = [
      "policy",
      "violation",
      "violates",
      "\u043D\u0430\u0440\u0443\u0448\u0435\u043D\u0438\u0435",
      "\u043F\u043E\u043B\u0438\u0442\u0438\u043A",
      "inappropriate",
      "harmful",
      "not allowed",
      "\u0437\u0430\u043F\u0440\u0435\u0449",
      "\u043D\u0435\u0434\u043E\u043F\u0443\u0441\u0442\u0438\u043C",
      "content policy",
      "safety",
      "\u0431\u0435\u0437\u043E\u043F\u0430\u0441\u043D\u043E\u0441\u0442",
      "blocked"
    ];
    const errors = document.querySelectorAll('[role="alert"], [class*="error"], [class*="Error"], [class*="warning"], [class*="Warning"]');
    for (const error of errors) {
      if (isVisible(error)) {
        const text = (error.textContent || "").toLowerCase();
        const isPolicyError = POLICY_KEYWORDS.some((kw) => text.includes(kw));
        if (isPolicyError) {
          status.hasError = true;
          status.isPolicyError = true;
          status.errorMessage = `Policy: ${error.textContent?.slice(0, 200)}`;
          break;
        }
        if (text.includes("error") || text.includes("failed") || text.includes("\u043E\u0448\u0438\u0431\u043A\u0430") || text.includes("\u043D\u0435 \u0443\u0434\u0430\u043B\u043E\u0441\u044C")) {
          status.hasError = true;
          status.errorMessage = error.textContent?.slice(0, 200);
          break;
        }
      }
    }
    return status;
  }
  var generationObserver = null;
  async function waitForGenerationComplete(prompt, slotId, timeoutMs = 5 * 60 * 1e3) {
    const startTime = Date.now();
    let checkCount = 0;
    let initialVideoCount = document.querySelectorAll("video").length;
    let initialImgCount = document.querySelectorAll('img[src*="googleusercontent"], img[src*="storage.googleapis"]').length;
    console.log(`[GenFlow] \u{1F3AC} Starting generation monitoring for prompt #${prompt.number}`);
    console.log(`[GenFlow] Initial state: ${initialVideoCount} videos, ${initialImgCount} relevant images`);
    if (generationCheckInterval) {
      clearInterval(generationCheckInterval);
      generationCheckInterval = null;
    }
    if (generationObserver) {
      generationObserver.disconnect();
      generationObserver = null;
    }
    return new Promise((resolve, reject) => {
      let resolved = false;
      const completeGeneration = (videoUrl) => {
        if (resolved)
          return;
        resolved = true;
        if (generationCheckInterval) {
          clearInterval(generationCheckInterval);
          generationCheckInterval = null;
        }
        if (generationObserver) {
          generationObserver.disconnect();
          generationObserver = null;
        }
        console.log(`[GenFlow] \u2705 Generation complete for prompt #${prompt.number}`);
        chrome.runtime.sendMessage({
          type: "GENERATION_COMPLETE",
          payload: {
            promptId: prompt.id,
            resultUrl: videoUrl || window.location.href
          }
        });
        resolve();
      };
      const failGeneration = (error) => {
        if (resolved)
          return;
        resolved = true;
        if (generationCheckInterval) {
          clearInterval(generationCheckInterval);
          generationCheckInterval = null;
        }
        if (generationObserver) {
          generationObserver.disconnect();
          generationObserver = null;
        }
        console.log(`[GenFlow] \u274C Generation failed: ${error}`);
        chrome.runtime.sendMessage({
          type: "GENERATION_FAILED",
          payload: {
            promptId: prompt.id,
            error
          }
        });
        reject(new Error(error));
      };
      generationObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
          for (const node of mutation.addedNodes) {
            if (node.nodeType !== Node.ELEMENT_NODE)
              continue;
            const el = node;
            if (el.tagName === "VIDEO") {
              const video = el;
              if (video.src || video.poster || video.querySelector("source")) {
                console.log("[GenFlow] MutationObserver: New video element detected!");
                setTimeout(() => {
                  const status = checkGenerationStatus();
                  if (status.isComplete) {
                    completeGeneration(status.videoUrl);
                  }
                }, 1e3);
              }
            }
            const nestedVideo = el.querySelector?.("video");
            if (nestedVideo) {
              console.log("[GenFlow] MutationObserver: Container with video added!");
              setTimeout(() => {
                const status = checkGenerationStatus();
                if (status.isComplete) {
                  completeGeneration(status.videoUrl);
                }
              }, 1e3);
            }
            const resultImg = el.querySelector?.('img[src*="googleusercontent"], img[src*="storage.googleapis"]');
            if (resultImg) {
              console.log("[GenFlow] MutationObserver: Result image detected!");
              setTimeout(() => {
                const status = checkGenerationStatus();
                if (status.isComplete) {
                  completeGeneration(status.videoUrl);
                }
              }, 1e3);
            }
          }
          if (mutation.type === "attributes" && mutation.attributeName === "src") {
            const target = mutation.target;
            if (target.tagName === "VIDEO" || target.tagName === "IMG") {
              console.log("[GenFlow] MutationObserver: Media src changed!");
              setTimeout(() => {
                const status = checkGenerationStatus();
                if (status.isComplete) {
                  completeGeneration(status.videoUrl);
                }
              }, 500);
            }
          }
        }
      });
      generationObserver.observe(document.body, {
        childList: true,
        subtree: true,
        attributes: true,
        attributeFilter: ["src", "poster"]
      });
      console.log("[GenFlow] MutationObserver started");
      generationCheckInterval = setInterval(() => {
        if (resolved)
          return;
        checkCount++;
        if (currentPromptId !== prompt.id) {
          console.log("[GenFlow] Prompt changed, stopping monitoring");
          failGeneration("Prompt changed");
          return;
        }
        const currentVideoCount = document.querySelectorAll("video").length;
        const currentImgCount = document.querySelectorAll('img[src*="googleusercontent"], img[src*="storage.googleapis"]').length;
        if (currentVideoCount > initialVideoCount || currentImgCount > initialImgCount) {
          console.log(`[GenFlow] Poll #${checkCount}: New media detected (videos: ${initialVideoCount}->${currentVideoCount}, imgs: ${initialImgCount}->${currentImgCount})`);
          const status2 = checkGenerationStatus();
          if (status2.isComplete) {
            completeGeneration(status2.videoUrl);
            return;
          }
        }
        const status = checkGenerationStatus();
        if (checkCount % 5 === 0) {
          console.log(`[GenFlow] Poll #${checkCount}: generating=${status.isGenerating}, complete=${status.isComplete}`);
        }
        if (status.isComplete) {
          completeGeneration(status.videoUrl);
          return;
        }
        if (status.hasError) {
          if (status.isPolicyError) {
            if (resolved)
              return;
            resolved = true;
            if (generationCheckInterval) {
              clearInterval(generationCheckInterval);
              generationCheckInterval = null;
            }
            if (generationObserver) {
              generationObserver.disconnect();
              generationObserver = null;
            }
            console.log(`[GenFlow] Policy error for prompt #${prompt.number}, skipping`);
            chrome.runtime.sendMessage({
              type: "POLICY_ERROR",
              payload: { promptId: prompt.id }
            });
            resolve();
            return;
          }
          failGeneration(status.errorMessage || "Generation failed");
          return;
        }
        if (Date.now() - startTime > timeoutMs) {
          failGeneration("Generation timed out (5 min)");
          return;
        }
      }, 2e3);
    });
  }
  async function handleFilmModeEdit(prompt, slotId) {
    console.log("[GenFlow] Film mode: Looking for Edit button...");
    const editButton = findEditButton();
    if (!editButton) {
      console.log("[GenFlow] Film mode: Edit button not found");
      return false;
    }
    console.log("[GenFlow] Film mode: Found Edit button, clicking...");
    editButton.click();
    await sleep(1e3);
    let editTextarea = null;
    for (let i = 0; i < 20; i++) {
      const textareas = document.querySelectorAll("textarea");
      for (const ta of textareas) {
        const placeholder = ta.getAttribute("placeholder") || "";
        if (placeholder.includes("\u0434\u043E\u0431\u0430\u0432\u0438\u0442\u044C") || placeholder.includes("add") || placeholder.includes("want to add")) {
          editTextarea = ta;
          break;
        }
      }
      const editVideoPreview = document.querySelector(".sc-179baefe-1 video[src], .sc-d97c7fd4-3 video[src]");
      if (editTextarea && editVideoPreview) {
        console.log("[GenFlow] Film mode: Edit dialog is ready");
        break;
      }
      const cancelBtn = findElementByText(["\u041E\u0442\u043C\u0435\u043D\u0430", "Cancel"], "button");
      if (cancelBtn && editTextarea) {
        console.log("[GenFlow] Film mode: Edit dialog detected (cancel button found)");
        break;
      }
      await sleep(300);
    }
    if (!editTextarea) {
      editTextarea = document.querySelector("#PINHOLE_TEXT_AREA_ELEMENT_ID");
    }
    if (!editTextarea) {
      console.log("[GenFlow] Film mode: Edit textarea not found");
      return false;
    }
    console.log("[GenFlow] Film mode: Entering prompt in edit textarea...");
    editTextarea.focus();
    await sleep(100);
    editTextarea.value = "";
    await sleep(50);
    const nativeTextAreaValueSetter = Object.getOwnPropertyDescriptor(
      window.HTMLTextAreaElement.prototype,
      "value"
    )?.set;
    if (nativeTextAreaValueSetter) {
      nativeTextAreaValueSetter.call(editTextarea, prompt.text);
    } else {
      editTextarea.value = prompt.text;
    }
    editTextarea.dispatchEvent(new Event("input", { bubbles: true }));
    editTextarea.dispatchEvent(new Event("change", { bubbles: true }));
    console.log(`[GenFlow] Film mode: Prompt entered: "${prompt.text.slice(0, 50)}..."`);
    await sleep(800);
    let createBtn = findCreateButton();
    for (let i = 0; i < 15; i++) {
      createBtn = findCreateButton();
      if (createBtn && !createBtn.disabled) {
        break;
      }
      await sleep(300);
    }
    if (!createBtn) {
      console.log("[GenFlow] Film mode: Create button not found in edit mode");
      return false;
    }
    if (createBtn.disabled) {
      console.log("[GenFlow] Film mode: Create button is disabled");
      return false;
    }
    console.log("[GenFlow] Film mode: Clicking Create button...");
    createBtn.click();
    console.log(`[GenFlow] Film mode: Started edit generation for prompt #${prompt.number}`);
    waitForGenerationComplete(prompt, slotId).catch((err) => {
      console.error("[GenFlow] Film mode generation monitoring failed:", err);
    });
    return true;
  }
  function findEditButton() {
    const buttons = document.querySelectorAll("button");
    for (const btn of buttons) {
      const icon = btn.querySelector("i");
      if (icon) {
        const iconText = icon.textContent?.trim() || "";
        if (iconText === "edit") {
          if (isVisible(btn)) {
            console.log("[GenFlow] Found Edit button by icon");
            return btn;
          }
        }
      }
    }
    for (const btn of buttons) {
      const span = btn.querySelector("span");
      const spanText = span?.textContent?.trim() || "";
      if (spanText === "\u0418\u0437\u043C\u0435\u043D\u0438\u0442\u044C" || spanText === "Edit") {
        if (isVisible(btn)) {
          console.log("[GenFlow] Found Edit button by text");
          return btn;
        }
      }
    }
    const videoControls = document.querySelector(".sc-7e665804-2, .sc-2b6ef9e5-0");
    if (videoControls) {
      const editBtn = videoControls.querySelector("button:has(i)");
      if (editBtn) {
        const icon = editBtn.querySelector("i");
        if (icon?.textContent?.includes("edit")) {
          console.log("[GenFlow] Found Edit button in video controls");
          return editBtn;
        }
      }
    }
    return null;
  }
  function clickFlowTabVideo() {
    const buttons = document.querySelectorAll("button");
    for (const btn of buttons) {
      const text = (btn.textContent || "").trim();
      if (text.includes("\u043F\u043E \u043E\u043F\u0438\u0441\u0430\u043D\u0438\u044E") || text.includes("by description") || text.includes("arrow_drop_down"))
        continue;
      if (text.length > 20)
        continue;
      if (FLOW_TEXTS.videoTab.some((t) => text.includes(t))) {
        console.log("[GenFlow] Clicking Video tab");
        btn.click();
        return true;
      }
    }
    return false;
  }
  function closeFlowDialogIfOpen() {
    document.body.dispatchEvent(new KeyboardEvent("keydown", { key: "Escape", keyCode: 27, bubbles: true }));
    const backdrop = document.querySelector('[data-state="open"] + [data-radix-portal], .backdrop, [class*="overlay"]');
    if (backdrop) {
      backdrop.click();
    }
  }
  async function injectPrompt(payload) {
    const { prompt, settings, slotId, generationMode } = payload;
    currentPromptId = prompt.id;
    try {
      if (isFlowHost()) {
        console.log(`[GenFlow] Starting Veo 3 generation for prompt #${prompt.number}`);
        console.log(`[GenFlow] Generation mode: ${generationMode || "single"}`);
        if (generationMode === "film" && prompt.imageUrl) {
          console.log("[GenFlow] Film mode detected - looking for Edit button...");
          const editHandled = await handleFilmModeEdit(prompt, slotId);
          if (editHandled) {
            return;
          }
          console.log("[GenFlow] Edit button not found, falling back to normal flow...");
        }
        await ensureFlowProjectPage();
        clickFlowTabVideo();
        await sleep(800);
        await selectFlowMode(settings.generationType);
        await sleep(500);
        if (settings.generationType === "image-to-video" && prompt.imageUrl) {
          await uploadImageForVideo(prompt.imageUrl, prompt.endImageUrl);
        }
        const textarea = findPromptTextarea();
        if (!textarea) {
          throw new Error("\u041D\u0435 \u043D\u0430\u0439\u0434\u0435\u043D\u043E \u043F\u043E\u043B\u0435 \u0434\u043B\u044F \u0432\u0432\u043E\u0434\u0430 \u043F\u0440\u043E\u043C\u043F\u0442\u0430. \u041F\u043E\u043F\u0440\u043E\u0431\u0443\u0439\u0442\u0435 \u043E\u0431\u043D\u043E\u0432\u0438\u0442\u044C \u0441\u0442\u0440\u0430\u043D\u0438\u0446\u0443.");
        }
        textarea.focus();
        textarea.value = "";
        await sleep(100);
        textarea.value = prompt.text;
        textarea.dispatchEvent(new Event("input", { bubbles: true }));
        textarea.dispatchEvent(new Event("change", { bubbles: true }));
        console.log(`[GenFlow] Prompt entered: "${prompt.text.slice(0, 50)}..."`);
        await sleep(600);
        closeFlowDialogIfOpen();
        await sleep(300);
        let createBtn = findCreateButton();
        console.log("[GenFlow] Waiting for Create button to become enabled...");
        for (let i = 0; i < 20; i++) {
          createBtn = findCreateButton();
          if (createBtn) {
            const isDisabled2 = createBtn.disabled || createBtn.getAttribute("aria-disabled") === "true" || createBtn.classList.contains("disabled");
            console.log(`[GenFlow] Create button check ${i + 1}: found=${!!createBtn}, disabled=${isDisabled2}`);
            if (!isDisabled2)
              break;
          } else {
            console.log(`[GenFlow] Create button check ${i + 1}: not found`);
          }
          await sleep(250);
        }
        if (!createBtn) {
          const allButtons = document.querySelectorAll("button");
          console.log(`[GenFlow] Available buttons (${allButtons.length}):`);
          allButtons.forEach((btn, i) => {
            const text = (btn.textContent || "").trim().substring(0, 30);
            if (text)
              console.log(`  ${i}: "${text}" disabled=${btn.disabled}`);
          });
          throw new Error("\u041A\u043D\u043E\u043F\u043A\u0430 \xAB\u0421\u043E\u0437\u0434\u0430\u0442\u044C\xBB \u043D\u0435 \u043D\u0430\u0439\u0434\u0435\u043D\u0430.");
        }
        const isDisabled = createBtn.disabled || createBtn.getAttribute("aria-disabled") === "true";
        if (isDisabled) {
          console.log("[GenFlow] Create button found but disabled. Checking prompt field...");
          const textarea2 = findPromptTextarea();
          console.log(`[GenFlow] Prompt textarea value: "${textarea2?.value?.substring(0, 50)}..."`);
          throw new Error("\u041A\u043D\u043E\u043F\u043A\u0430 \xAB\u0421\u043E\u0437\u0434\u0430\u0442\u044C\xBB \u043D\u0435\u0434\u043E\u0441\u0442\u0443\u043F\u043D\u0430. \u041F\u0440\u043E\u0432\u0435\u0440\u044C\u0442\u0435, \u0447\u0442\u043E \u043F\u0440\u043E\u043C\u043F\u0442 \u0432\u0432\u0435\u0434\u0451\u043D \u043A\u043E\u0440\u0440\u0435\u043A\u0442\u043D\u043E.");
        }
        console.log("[GenFlow] Clicking Create button...");
        createBtn.click();
        console.log(`[GenFlow] Veo 3: Started ${settings.generationType} generation for prompt #${prompt.number}`);
        waitForGenerationComplete(prompt, slotId).catch((err) => {
          console.error("[GenFlow] Generation monitoring failed:", err);
        });
      } else {
        await waitForElement('textarea, [contenteditable="true"]', 15e3);
        await sleep(1e3);
        const inputEl = findPromptInput();
        if (!inputEl) {
          console.log("[GenFlow] Page HTML:", document.body.innerHTML.slice(0, 2e3));
          throw new Error("Could not find prompt input field");
        }
        if (inputEl.tagName === "TEXTAREA") {
          inputEl.value = "";
          inputEl.value = prompt.text;
          inputEl.dispatchEvent(new Event("input", { bubbles: true }));
          inputEl.dispatchEvent(new Event("change", { bubbles: true }));
        } else {
          inputEl.innerHTML = "";
          inputEl.textContent = prompt.text;
          inputEl.dispatchEvent(new InputEvent("input", { bubbles: true, data: prompt.text }));
        }
        inputEl.focus();
        await applySettings(settings);
        await sleep(800);
        const generateButton = findGenerateButton();
        if (!generateButton) {
          console.log("[GenFlow] All buttons on page:");
          document.querySelectorAll("button").forEach((btn, i) => {
            console.log(`  ${i}: "${btn.textContent?.trim()}" class="${btn.className}"`);
          });
          throw new Error("Could not find generate button");
        }
        generateButton.click();
        console.log(`[GenFlow] Veo 3 (AI Studio): Started generation for prompt #${prompt.number}`);
        waitForGenerationComplete(prompt, slotId).catch((err) => {
          console.error("[GenFlow] AI Studio generation monitoring failed:", err);
        });
      }
    } catch (error) {
      console.error("[GenFlow] Veo injection failed:", error);
      chrome.runtime.sendMessage({
        type: "GENERATION_FAILED",
        payload: {
          promptId: prompt.id,
          error: error instanceof Error ? error.message : "Unknown error"
        }
      });
    }
  }
  async function applySettings(settings) {
    const qualitySelectors = document.querySelectorAll('[role="listbox"], select');
    for (const selector of qualitySelectors) {
      const text = selector.textContent?.toLowerCase() || "";
      if (text.includes("quality") || text.includes("resolution")) {
        const options = selector.querySelectorAll('[role="option"], option');
        for (const option of options) {
          const optionText = option.textContent?.toLowerCase() || "";
          if (optionText.includes(settings.quality.replace("p", ""))) {
            option.click();
            break;
          }
        }
      }
    }
    const ratioButtons = document.querySelectorAll('button, [role="radio"]');
    for (const button of ratioButtons) {
      const text = button.textContent || "";
      if (text.includes(settings.aspectRatio)) {
        button.click();
        break;
      }
    }
    for (const button of ratioButtons) {
      const text = button.textContent?.toLowerCase() || "";
      const targetModel = settings.model === "veo3.1-fast" ? "fast" : "quality";
      if (text.includes(targetModel)) {
        button.click();
        break;
      }
    }
  }
  function findPromptInput() {
    const textarea = findPromptTextarea();
    if (textarea)
      return textarea;
    const contentEditable = document.querySelector('[contenteditable="true"]');
    if (contentEditable && isVisible(contentEditable))
      return contentEditable;
    return null;
  }
  function findGenerateButton() {
    const selectors = [
      // AI Studio run/generate buttons
      'button[aria-label*="Run" i]',
      'button[aria-label*="Generate" i]',
      'button[aria-label*="Send" i]',
      'button[aria-label*="Submit" i]',
      'button[data-testid*="run"]',
      'button[data-testid*="generate"]',
      'button[data-testid*="send"]',
      // Material Design buttons
      "button.mdc-button--raised",
      "button.mat-raised-button",
      "button.mat-flat-button",
      // Generic
      'button[type="submit"]',
      ".generate-btn",
      "button.primary"
    ];
    for (const selector of selectors) {
      try {
        const elements = document.querySelectorAll(selector);
        for (const el of elements) {
          const btn = el;
          if (!btn.disabled && isVisible(btn)) {
            console.log(`[GenFlow] Found generate button with selector: ${selector}`);
            return btn;
          }
        }
      } catch {
      }
    }
    const buttons = document.querySelectorAll("button");
    for (const button of buttons) {
      const text = button.textContent?.toLowerCase().trim() || "";
      const ariaLabel = button.getAttribute("aria-label")?.toLowerCase() || "";
      const keywords = ["generate", "run", "create", "send", "submit", "go"];
      const hasKeyword = keywords.some((k) => text.includes(k) || ariaLabel.includes(k));
      if (hasKeyword && !button.disabled && isVisible(button)) {
        console.log(`[GenFlow] Found button by text: "${text}"`);
        return button;
      }
    }
    for (const button of buttons) {
      if (!button.disabled && isVisible(button)) {
        const svg = button.querySelector("svg");
        if (svg) {
          const paths = svg.querySelectorAll("path");
          for (const path of paths) {
            const d = path.getAttribute("d") || "";
            if (d.includes("M2") && d.includes("l") || d.includes("arrow") || d.includes("send")) {
              console.log("[GenFlow] Found icon button with send/arrow");
              return button;
            }
          }
        }
      }
    }
    return null;
  }
  function waitForElement(selector, timeout) {
    return new Promise((resolve, reject) => {
      const element = document.querySelector(selector);
      if (element) {
        resolve(element);
        return;
      }
      const observer = new MutationObserver(() => {
        const element2 = document.querySelector(selector);
        if (element2) {
          observer.disconnect();
          resolve(element2);
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      setTimeout(() => {
        observer.disconnect();
        reject(new Error(`Element ${selector} not found within ${timeout}ms`));
      }, timeout);
    });
  }
  function sleep(ms) {
    return new Promise((resolve) => setTimeout(resolve, ms));
  }
  console.log("[GenFlow] Veo content script loaded for:", window.location.hostname);
  console.log("[GenFlow] Veo content script URL:", window.location.href);
  if (window.location.href.includes("labs.google")) {
    setTimeout(() => {
      console.log("[GenFlow] Page videos:", document.querySelectorAll("video").length);
      console.log("[GenFlow] Page images:", document.querySelectorAll("img").length);
      console.log("[GenFlow] Page buttons:", document.querySelectorAll("button").length);
      document.querySelectorAll("video").forEach((v, i) => {
        const video = v;
        console.log(`[GenFlow] Video ${i}: src=${video.src}, poster=${video.poster}, visible=${isVisible(video)}`);
      });
    }, 3e3);
  }
})();
