"use strict";
(() => {
  // src/content/whisk.ts
  var currentPromptId = null;
  var observer = null;
  var WHISK = {
    promptTextarea: [
      "#nameInput",
      'textarea[placeholder*="Describe your idea"]',
      'textarea[placeholder*="roll the dice"]',
      "textarea.sc-da7d3cfd-2",
      "textarea"
    ],
    submitButton: [
      'button[aria-label="Submit prompt"]',
      'button[type="submit"]',
      "button.sc-bece3008-0"
    ],
    addImagesButton: 'button:has(span:contains("Add Images")), button.sc-63569c0e-0'
  };
  console.log("[GenFlow] Whisk content script loaded:", window.location.href);
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    console.log("[GenFlow] Whisk received message:", message.type);
    if (message.type === "PING") {
      sendResponse({ pong: true });
      return false;
    }
    if (message.type === "INJECT_PROMPT") {
      const payload = message.payload;
      if (payload.settings?.service !== "whisk")
        return false;
      injectPrompt(payload);
      sendResponse({ success: true });
      return false;
    } else if (message.type === "UPLOAD_CONTEXT") {
      uploadContext(message.payload.context);
      sendResponse({ success: true });
      return false;
    }
    return false;
  });
  function isWhiskHost() {
    return /labs\.google(\.com)?\/fx\/tools\/whisk/i.test(window.location.href);
  }
  function sleep(ms) {
    return new Promise((r) => setTimeout(r, ms));
  }
  async function simulateTyping(element, text) {
    element.focus();
    element.value = "";
    element.dispatchEvent(new Event("input", { bubbles: true }));
    const charsToType = Math.min(5, text.length);
    for (let i = 0; i < charsToType; i++) {
      const char = text[i];
      element.dispatchEvent(new KeyboardEvent("keydown", {
        key: char,
        code: `Key${char.toUpperCase()}`,
        bubbles: true
      }));
      element.value = text.substring(0, i + 1);
      element.dispatchEvent(new InputEvent("input", {
        bubbles: true,
        cancelable: true,
        inputType: "insertText",
        data: char
      }));
      element.dispatchEvent(new KeyboardEvent("keyup", {
        key: char,
        code: `Key${char.toUpperCase()}`,
        bubbles: true
      }));
      await sleep(20);
    }
    if (text.length > charsToType) {
      const nativeValueSetter = Object.getOwnPropertyDescriptor(
        window.HTMLTextAreaElement.prototype,
        "value"
      )?.set;
      if (nativeValueSetter) {
        nativeValueSetter.call(element, text);
      } else {
        element.value = text;
      }
      element.dispatchEvent(new InputEvent("input", {
        bubbles: true,
        cancelable: true,
        inputType: "insertText",
        data: text.substring(charsToType)
      }));
    }
    element.dispatchEvent(new Event("change", { bubbles: true }));
  }
  function isVisible(el) {
    const style = window.getComputedStyle(el);
    return style.display !== "none" && style.visibility !== "hidden" && style.opacity !== "0" && el.offsetWidth > 0 && el.offsetHeight > 0;
  }
  function findPromptInput() {
    for (const sel of WHISK.promptTextarea) {
      const el = document.querySelector(sel);
      if (el && el instanceof HTMLTextAreaElement && isVisible(el)) {
        console.log(`[GenFlow] Whisk: Found prompt input with: ${sel}`);
        return el;
      }
    }
    console.log("[GenFlow] Whisk: Prompt input not found");
    return null;
  }
  function findSubmitButton() {
    for (const sel of WHISK.submitButton) {
      const el = document.querySelector(sel);
      if (el && isVisible(el)) {
        console.log(`[GenFlow] Whisk: Found submit button with: ${sel}`);
        return el;
      }
    }
    const buttons = document.querySelectorAll("button");
    for (const btn of buttons) {
      const icon = btn.querySelector("i");
      if (icon && icon.textContent?.includes("arrow_forward")) {
        console.log("[GenFlow] Whisk: Found submit button by arrow icon");
        return btn;
      }
    }
    console.log("[GenFlow] Whisk: Submit button not found");
    return null;
  }
  async function waitForPageReady() {
    console.log("[GenFlow] Whisk: Waiting for page to be ready...");
    for (let i = 0; i < 30; i++) {
      const textarea = findPromptInput();
      if (textarea) {
        console.log("[GenFlow] Whisk: Page is ready");
        return;
      }
      await sleep(500);
    }
    throw new Error("\u0421\u0442\u0440\u0430\u043D\u0438\u0446\u0430 Whisk \u043D\u0435 \u0437\u0430\u0433\u0440\u0443\u0437\u0438\u043B\u0430\u0441\u044C. \u041F\u043E\u043F\u0440\u043E\u0431\u0443\u0439\u0442\u0435 \u043E\u0431\u043D\u043E\u0432\u0438\u0442\u044C \u0441\u0442\u0440\u0430\u043D\u0438\u0446\u0443.");
  }
  async function injectPrompt(payload) {
    const { prompt, settings, slotId, context, generationMode } = payload;
    currentPromptId = prompt.id;
    try {
      console.log(`[GenFlow] Whisk: Starting injection for prompt #${prompt.number}`);
      console.log(`[GenFlow] Whisk: Current URL: ${window.location.href}`);
      console.log(`[GenFlow] Whisk: Generation mode: ${generationMode || "single"}`);
      if (prompt.imageUrl) {
        console.log(`[GenFlow] Whisk: Has input image URL: ${prompt.imageUrl.substring(0, 80)}...`);
      }
      if (!isWhiskHost()) {
        throw new Error("Whisk content script should run on labs.google/fx/tools/whisk");
      }
      await waitForPageReady();
      await sleep(1e3);
      if (context && (context.objects?.length || context.characters?.length || context.locations?.length)) {
        console.log("[GenFlow] Whisk: Uploading context images...");
        await uploadContext(context);
        await sleep(500);
      }
      const filmDialog = document.querySelector('div[role="dialog"][data-state="open"]');
      const filmTextarea = document.querySelector("textarea#prompt-editor, textarea.sc-fb72904e-12");
      if (prompt.imageUrl && filmDialog && filmTextarea && isVisible(filmDialog)) {
        console.log("[GenFlow] Whisk: Film mode - found edit dialog, entering prompt there...");
        await handleFilmModeDialog(filmTextarea, prompt.text);
        const dialogGenerateBtn = filmDialog.querySelector('button.sc-fb72904e-18, button:has(div[data-type="button-overlay"])');
        if (dialogGenerateBtn) {
          console.log("[GenFlow] Whisk: Clicking Generate in dialog...");
          dialogGenerateBtn.click();
          startCompletionMonitor(prompt, slotId);
          return;
        }
      }
      const textarea = findPromptInput();
      if (!textarea) {
        throw new Error("\u041D\u0435 \u043D\u0430\u0439\u0434\u0435\u043D\u043E \u043F\u043E\u043B\u0435 \u0434\u043B\u044F \u0432\u0432\u043E\u0434\u0430 \u043F\u0440\u043E\u043C\u043F\u0442\u0430");
      }
      console.log(`[GenFlow] Whisk: Entering prompt text...`);
      await simulateTyping(textarea, prompt.text);
      console.log(`[GenFlow] Whisk: Prompt entered: "${prompt.text.slice(0, 50)}..."`);
      console.log(`[GenFlow] Whisk: Textarea value after input: "${textarea.value.slice(0, 50)}..."`);
      await sleep(800);
      let submitBtn = findSubmitButton();
      console.log("[GenFlow] Whisk: Waiting for Submit button to be enabled...");
      for (let i = 0; i < 30; i++) {
        submitBtn = findSubmitButton();
        if (submitBtn) {
          const isDisabled = submitBtn.disabled;
          console.log(`[GenFlow] Whisk: Submit button check ${i + 1}/30: disabled=${isDisabled}`);
          if (!isDisabled) {
            console.log("[GenFlow] Whisk: Submit button is now enabled!");
            break;
          }
        }
        if (i % 5 === 4) {
          const ta = findPromptInput();
          if (ta) {
            ta.dispatchEvent(new Event("input", { bubbles: true }));
            ta.dispatchEvent(new Event("change", { bubbles: true }));
          }
        }
        await sleep(200);
      }
      if (!submitBtn) {
        throw new Error("\u041A\u043D\u043E\u043F\u043A\u0430 \u043E\u0442\u043F\u0440\u0430\u0432\u043A\u0438 \u043D\u0435 \u043D\u0430\u0439\u0434\u0435\u043D\u0430");
      }
      if (submitBtn.disabled) {
        console.log("[GenFlow] Whisk: Submit button still disabled after waiting.");
        console.log("[GenFlow] Whisk: Textarea value:", findPromptInput()?.value?.slice(0, 50));
        console.log("[GenFlow] Whisk: Attempting click on disabled button...");
        submitBtn.click();
        await sleep(300);
        submitBtn = findSubmitButton();
        if (submitBtn && submitBtn.disabled) {
          throw new Error("\u041A\u043D\u043E\u043F\u043A\u0430 \u043E\u0442\u043F\u0440\u0430\u0432\u043A\u0438 \u043D\u0435\u0434\u043E\u0441\u0442\u0443\u043F\u043D\u0430. \u0412\u043E\u0437\u043C\u043E\u0436\u043D\u043E, \u043D\u0443\u0436\u043D\u043E \u0434\u043E\u0431\u0430\u0432\u0438\u0442\u044C \u0438\u0437\u043E\u0431\u0440\u0430\u0436\u0435\u043D\u0438\u0435.");
        }
      } else {
        console.log("[GenFlow] Whisk: Clicking Submit button...");
        submitBtn.click();
      }
      console.log(`[GenFlow] Whisk: Started generation for prompt #${prompt.number}`);
      startCompletionMonitor(prompt, slotId);
    } catch (error) {
      console.error("[GenFlow] Whisk injection failed:", error);
      chrome.runtime.sendMessage({
        type: "GENERATION_FAILED",
        payload: {
          promptId: prompt.id,
          error: error instanceof Error ? error.message : "Unknown error"
        }
      });
    }
  }
  async function uploadContext(context) {
    const addImagesBtn = document.querySelector("button.sc-63569c0e-0");
    if (addImagesBtn && isVisible(addImagesBtn)) {
      console.log("[GenFlow] Whisk: Clicking Add Images button...");
      addImagesBtn.click();
      await sleep(800);
    }
    for (const obj of context.objects || []) {
      await uploadImage(obj.imageUrl, "object");
    }
    for (const char of context.characters || []) {
      await uploadImage(char.imageUrl, "subject");
    }
    for (const loc of context.locations || []) {
      await uploadImage(loc.imageUrl, "scene");
    }
  }
  async function handleFilmModeDialog(textarea, promptText) {
    console.log("[GenFlow] Whisk: Entering prompt in Film mode dialog...");
    textarea.focus();
    await sleep(100);
    textarea.value = "";
    textarea.dispatchEvent(new Event("input", { bubbles: true }));
    await sleep(50);
    await simulateTyping(textarea, promptText);
    console.log(`[GenFlow] Whisk: Film dialog prompt entered: "${promptText.slice(0, 50)}..."`);
    await sleep(300);
  }
  async function uploadImage(imageUrl, _type) {
    try {
      const fileInput = document.querySelector('input[type="file"]');
      if (fileInput && imageUrl.startsWith("data:")) {
        const response = await fetch(imageUrl);
        const blob = await response.blob();
        const file = new File([blob], `image_${Date.now()}.png`, { type: "image/png" });
        const dataTransfer = new DataTransfer();
        dataTransfer.items.add(file);
        fileInput.files = dataTransfer.files;
        fileInput.dispatchEvent(new Event("change", { bubbles: true }));
        console.log("[GenFlow] Whisk: Image uploaded");
      }
      await sleep(1e3);
    } catch (e) {
      console.warn(`[GenFlow] Whisk: Failed to upload image:`, e);
    }
  }
  function startCompletionMonitor(prompt, slotId) {
    if (observer)
      observer.disconnect();
    const startTime = Date.now();
    const timeout = 3 * 60 * 1e3;
    let checkCount = 0;
    let lastDialogState = false;
    console.log("[GenFlow] Whisk: Starting completion monitor...");
    const checkCompletion = () => {
      checkCount++;
      if (currentPromptId !== prompt.id) {
        observer?.disconnect();
        return;
      }
      const resultDialog = document.querySelector('div[role="dialog"][data-state="open"]');
      const dialogImage = resultDialog?.querySelector('img[src^="blob:"], img[src*="http"]');
      const promptEditor = resultDialog?.querySelector("textarea#prompt-editor");
      if (resultDialog && dialogImage && promptEditor && isVisible(resultDialog)) {
        const dialogJustAppeared = !lastDialogState;
        lastDialogState = true;
        if (dialogJustAppeared) {
          const imgSrc = dialogImage.src;
          observer?.disconnect();
          console.log(`[GenFlow] Whisk: \u2705 Generation complete for prompt #${prompt.number}`);
          console.log(`[GenFlow] Whisk: Result dialog appeared with image: ${imgSrc.substring(0, 80)}`);
          chrome.runtime.sendMessage({
            type: "GENERATION_COMPLETE",
            payload: { promptId: prompt.id, resultUrl: imgSrc }
          });
          return;
        }
      } else {
        lastDialogState = false;
      }
      const imgs = document.querySelectorAll('img[src*="http"]:not([src*="googleusercontent"])');
      for (const img of imgs) {
        const imgEl = img;
        const src = imgEl.src;
        if (!src || src.includes("profile") || src.includes("avatar"))
          continue;
        if (imgEl.closest('div[role="dialog"]'))
          continue;
        const w = imgEl.naturalWidth || imgEl.clientWidth;
        const h = imgEl.naturalHeight || imgEl.clientHeight;
        if (w > 256 && h > 256) {
          const rect = imgEl.getBoundingClientRect();
          if (rect.width > 200 && rect.height > 200) {
            observer?.disconnect();
            console.log(`[GenFlow] Whisk: \u2705 Generation complete (large image found)`);
            console.log(`[GenFlow] Whisk: Result URL: ${src.substring(0, 100)}`);
            chrome.runtime.sendMessage({
              type: "GENERATION_COMPLETE",
              payload: { promptId: prompt.id, resultUrl: src }
            });
            return;
          }
        }
      }
      const downloadBtn = document.querySelector('button[aria-label*="download" i], a[download]');
      if (downloadBtn && isVisible(downloadBtn)) {
        const href = downloadBtn.getAttribute("href") || "";
        observer?.disconnect();
        console.log("[GenFlow] Whisk: \u2705 Download button found, generation complete");
        chrome.runtime.sendMessage({
          type: "GENERATION_COMPLETE",
          payload: {
            promptId: prompt.id,
            resultUrl: href || window.location.href
          }
        });
        return;
      }
      if (Date.now() - startTime > timeout) {
        observer?.disconnect();
        console.log("[GenFlow] Whisk: \u23F0 Timeout");
        chrome.runtime.sendMessage({
          type: "GENERATION_FAILED",
          payload: { promptId: prompt.id, error: "Generation timed out (3 min)" }
        });
      }
    };
    observer = new MutationObserver(checkCompletion);
    observer.observe(document.body, { childList: true, subtree: true, attributes: true });
    const intervalId = setInterval(() => {
      if (currentPromptId !== prompt.id) {
        clearInterval(intervalId);
        return;
      }
      checkCompletion();
    }, 2e3);
  }
  console.log("[GenFlow] Whisk content script ready");
})();
